<?php

namespace App\Http\Controllers\Dashboard\Contests;
use App\Http\Controllers\Controller;
use App\Models\Contest;
use App\Models\ContestAnswer;
use App\Models\ContestQuestion;
use App\Models\illustrationContest;
use App\Models\Level;
use App\Models\Contestmodel;
use App\Models\IllustrationContestQuestion;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class ContestController extends Controller
{
    public function index(Request $request)
    {
        return view('dashboard.contests.index', [
            'modal' => $request->modal,
            'levels' => Level::all(),
        ]);
    }

    public function getIndex(Request $request)
    {
        $query = Contest::query()
            ->when($request->search['value'] ?? null, fn($q, $search) => $q->where('name', 'like', "%$search%"))
            ->when($request->search_level_id ?? null, function ($q, $levelId) {
                if ($levelId != -1) $q->where('level_id', $levelId);
            });

        if ($request->has('order')) {
            $columns = ['id', 'name'];
            $index = $request->input('order.0.column');
            $direction = $request->input('order.0.dir');
            if (isset($columns[$index])) $query->orderBy($columns[$index], $direction);
        }

        return DataTables::of($query)
            ->addColumn('name', fn($data) => $data->name)
            ->addColumn('levels', fn($data) => $data->levels->name ?? null)
            ->addColumn('actions', fn($data) => view('dashboard.contests.partials.actions', compact('data')))
            ->rawColumns(['actions'])
            ->make(true);
    }

    public function store(Request $request)
    {
        $data = $request->except(['_token','contest_id']);
        if ($request->photo) {
            $level = Level::findOrFail($request->level_id);
            $data['photo'] = $request->photo->store("{$level->code}/contests", 'public');
        }
        $data['question_count'] = 0;
        $data['is_random'] = $request->is_random?1:0;

        Contest::create($data);

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }

    public function update(Request $request)
    {
        $contest = Contest::findOrFail($request->contest_id);
        if ($request->photo) {
            $level = Level::findOrFail($request->level_id);
            $data['photo'] = $request->photo->store("{$level->code}/contests", 'public');
            $contest->update(['photo' => $data['photo']]);
        }
        $data['is_random'] = $request->is_random?1:0;
        $contest->update($request->except(['_token', 'contest_id','photo']));
        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }

    public function questionList(Request $request)
    {
        $questions = ContestQuestion::with('answers')
            ->where('contest_id', $request->contest_id)
            ->get()
            ->map(function ($question) {
                $question->attachment = $question->getAttachments();
                return $question;
            });

        return response()->json([
            'success' => true,
            'questions' => $questions,
        ]);
    }


    public function getAttachments(Request $request){
        $attachments = Contestmodel::query()->where('contest_id',$request->contest_id)->get();
        return response()->json(['attachments' => $attachments]);
    }
    public function saveAttachment(Request $request)
    {
        $contest = Contest::findOrFail($request->contest_id);
        $attachment = $request->attachment_book->store("contests/{$contest->id}", 'public');

        Contestmodel::create([
            'contest_id' => $contest->id,
            'pdf_url' => $attachment,
            'title' => $request->title,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }

    public function getQuestions(Request $request){
        $contestId = $request->input('contest_id');

        // Get the questions related to the contest
        $questions = ContestQuestion::where('contest_id', $contestId)->get();

        // Return the questions as JSON
        return response()->json(['questions' => $questions]);
        }



    public function addQuestion(Request $request)
    {
        $contest = Contest::findOrFail($request->contest_id);
        $question = ContestQuestion::create([
            'contest_id' => $contest->id,
            'name' => $request->name,
            'questions' => $request->question,
            'mark' => $request->mark,
        ]);

        foreach ($request->answers as $answerData) {
            if ($answerData['answer']) {
                ContestAnswer::create([
                    'answer' => $answerData['answer'],
                    'question_contest_id' => $question->id,
                    'is_correct' =>  isset($answerData['is_correct']) ?1: 0,
                ]);
            }
        }

        $correctAnswer = $question->answers()->where('is_correct', 1)->first();
        $question->update(['is_correct' => $correctAnswer?->id]);

        if ($request->question_attachment) {
            $attachment = $request->question_attachment->store("contests/{$contest->id}", 'public');
            $question->update(['attachment' => $attachment]);
        }

        $contest->update(['question_count' => $contest->questions()->count()]);

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }

    public function delete(Request $request)
    {
        Contest::findOrFail($request->id)->delete();

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }

    public function addPhoto(Request $request)
    {
        $request->validate([
            'contest_id' => 'required|exists:contests,id',
            'question_id' => 'required|array',
            'attachment_photo' => 'required|file',
        ]);

        $contest = Contest::findOrFail($request->contest_id);
        $photoPath = $request->attachment_photo->store("contests/{$contest->id}", 'public');

        $illustration = illustrationContest::query()->create(
            ['contest_id' => $contest->id,
            'photo' => $photoPath]
        );


        foreach($request->question_id as $value){

            IllustrationContestQuestion::query()->create([
                'question_id'=>$value,
                'contest_id'=>$contest->id,
                'illustration_contest_id'=>$illustration->id

            ]);
        // $illustration->questions()->sync(
        //     collect($request->question_id)->mapWithKeys(
        //         fn($id) => [$id => ['contest_id' => $contest->id]]
        //     )
        // );
    }

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح',
        ]);
    }
}
