<?php

namespace App\Http\Controllers\Dashboard\Levels;

use App\Http\Controllers\Controller;
use App\Models\ClassRoom;
use App\Models\Level;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class LevelController extends Controller
{
    public function index(Request $request)
    {
        // Include the modal for view if passed in the request
        $data['modal'] = $request->modal;
        $data['classRooms'] = ClassRoom::all(); // Get all classrooms

        return view('dashboard.levels.index', $data); // Return view with data
    }

    public function getIndex(Request $request)
    {
        // Apply search filter if provided
        $search = $request->search['value'] ?? false;
        $query = Level::query()
            ->when($search, function ($q) use ($search) {
                $q->where('name', 'like', '%' . $search . '%');
            });

        // Handle sorting based on DataTables columns
        if ($request->has('order')) {
            $columnIndex = $request->input('order.0.column');
            $direction = $request->input('order.0.dir');
            $columns = ['id', 'name'];

            // Apply sorting on the specified column
            if (isset($columns[$columnIndex])) {
                $query->orderBy($columns[$columnIndex], $direction);
            }
        }

        // Return the data for DataTables
        return DataTables::of($query)
            ->addColumn('actions', function ($data) {
                return view('dashboard.levels.partials.actions', compact('data')); // Render actions view
            })
            ->rawColumns(['actions']) // Ensure actions are rendered as raw HTML
            ->make(true);
    }

    public function store(Request $request)
    {
        // Validate the incoming request
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:255',
        ]);

        // Create a new Level
        Level::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح', // Success message
        ]);
    }

    public function update(Request $request)
    {
        // Validate the incoming request
        $validated = $request->validate([
            'level_id' => 'required|exists:levels,id',
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:255',
            'class_room_id' => 'array|exists:class_rooms,id',
        ]);

        // Find the level by ID and update
        $level = Level::findOrFail($request->level_id);

        // Sync classrooms with the level
        $level->classRooms()->sync($request->class_room_id);

        // Update the level details
        $level->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح', // Success message
        ]);
    }

    public function delete(Request $request)
    {
        // Validate incoming request
        $validated = $request->validate([
            'id' => 'required|exists:levels,id',
        ]);

        // Delete the level
        Level::findOrFail($validated['id'])->delete();

        return response()->json([
            'success' => true,
            'message' => 'تمت العملية بنجاح', // Success message
        ]);
    }
}
