<?php

namespace App\Imports;

use App\Models\Contest;
use App\Models\ContestAnswer;
use App\Models\ContestQuestion;
use Illuminate\Support\Facades\Log; // Log errors for debugging
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;

class ContestQuestionsImportExeclSheet implements ToModel, WithValidation, WithHeadingRow, SkipsOnError, SkipsOnFailure
{
    use SkipsErrors, SkipsFailures;

    public $failedRows = []; // Array to store failed rows

    protected $errors = [];

    protected $attributeToColumnMap = [
        'contest_question_name' => 'Contest Question Name',
        'contest_name' => 'Contest Name',
        'correct_answer' => 'Correct Answer',
    ];

    /**
     * Handle the model import
     *
     * @param array $row
     * @return ContestAnswer|null
     */
    public function model(array $row)
    {
        try {
            // Find contest by name
            $contest = Contest::query()->where('name', $row['contest_name'])->first();
            if (!$contest) {
                throw new \Exception("Contest not found: " . $row['contest_name']);
            }

            // Find the contest question
            $contest_question = ContestQuestion::query()
                ->where('questions', $row['contest_question_name'])
                ->first();

            if (!$contest_question) {
                throw new \Exception("Contest Question not found: " . $row['contest_question_name']);
            }

            // Log the contest question and answer for debugging
            Log::info("Contest: {$contest->name}, Question: {$row['contest_question_name']}, Correct Answer: {$row['correct_answer']}");

            // Reset the previous correct answer (if any) for this question
            ContestAnswer::where('question_contest_id', $contest_question->id)
                ->where('is_correct', 1)
                ->update(['is_correct' => 0]);

            // Update or create the correct answer for the contest question
            ContestAnswer::updateOrCreate(
                [
                    'question_contest_id' => $contest_question->id,
                    'answer' => $row['correct_answer'],
                ],
                [
                    'answer' => $row['correct_answer'],
                    'is_correct' => 1, // Mark the answer as correct
                ]
            );

            // Find the correct answer and update the contest question
            $correctAnswer = $contest_question->answers()->where('is_correct', 1)->first();
            if ($correctAnswer) {
                $contest_question->update(['is_correct' => $correctAnswer->id]);
            } else {
                Log::warning("No correct answer found for question: {$row['contest_question_name']}");
            }

            return null; // Nothing needs to be returned as we are updating the database

        } catch (\Exception $e) {
            Log::error('Error importing contest question: ' . $e->getMessage());
            return null; // In case of error, just return null
        }
    }

    /**
     * Define validation rules for each field.
     *
     * @return array
     */
    public function rules(): array
    {
        return [
            'contest_name' => ['required', function ($attribute, $value, $fail) {
                if (!Contest::query()->where('name', $value)->exists()) {
                    $fail("The Contest '{$value}' does not exist.");
                }
            }],
            'contest_question_name' => 'required',
            'correct_answer' => 'required',
        ];
    }

    /**
     * Handle failures during the import process.
     *
     * @param \Maatwebsite\Excel\Validators\Failure ...$failures
     */
    public function onFailure(\Maatwebsite\Excel\Validators\Failure ...$failures)
    {
        foreach ($failures as $failure) {
            $rowIndex = $failure->row();
            if (!isset($this->errors[$rowIndex])) {
                $this->errors[$rowIndex] = [
                    'row' => $rowIndex,
                    'values' => $failure->values(),
                    'errors' => []
                ];
            }

            foreach ($failure->errors() as $error) {
                if (!in_array($error, $this->errors[$rowIndex]['errors'])) {
                    $this->errors[$rowIndex]['errors'][] = $error;
                }
            }
        }
    }

    /**
     * Get the error details for all failed rows.
     *
     * @return array
     */
    public function getErrors()
    {
        return $this->errors;
    }
}
