<?php

namespace App\Livewire\ClassSections;

use App\Models\ClassRoom;
use App\Models\ClassSection;
use App\Models\ClassPeriod;
use Livewire\Component;
use Livewire\WithPagination;

class ClassSectionComponent extends Component
{
    use WithPagination;

    public $search = '';
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $isModalOpen = false;
    public $isDeleteConfirmationOpen = false;
    public $isCalendarModalOpen = false; // Calendar modal state
    public $deleteConfirmation = false;
    public $classSectionId;
    public $name;
    public $status = 1;
    public $capacity;
    public $note;
    public $class_room_id = null;
    public $classRooms = [];
    public $classPeriods = []; // Store class periods for calendar

    protected $rules = [
        'name' => 'required|string|max:255',
        'capacity' => 'required|integer|min:0',
        'class_room_id' => 'required|exists:class_rooms,id',
    ];

    public function openModal($id = null)
    {
        $this->reset(['name', 'capacity', 'classSectionId', 'status', 'class_room_id']);
        $this->resetValidation();

        $this->classRooms = ClassRoom::all();

        if ($id) {
            $this->loadClassSection($id);
        }

        $this->isModalOpen = true;
    }

    private function loadClassSection($id)
    {
        $classSection = ClassSection::findOrFail($id);
        $this->classSectionId = $id;
        $this->name = $classSection->name;
        $this->capacity = $classSection->capacity;
        $this->status = $classSection->status;
        $this->class_room_id = $classSection->class_room_id;
    }

    public function closeModal()
    {
        $this->isModalOpen = false;
    }

    public function save()
    {
        $this->validate();

        ClassSection::updateOrCreate(
            ['id' => $this->classSectionId],
            [
                'name' => $this->name,
                'capacity' => $this->capacity,
                'class_room_id' => $this->class_room_id,
                'status' => $this->status ? 1 : 0,
            ]
        );

        $this->dispatch('success', ['message' => 'تم تنفيذ العملية بنجاح']);
        $this->closeModal();
    }

    public function confirmDelete($id)
    {
        $this->classSectionId = $id;
        $this->isDeleteConfirmationOpen = true;
    }

    public function closeDeleteConfirmation()
    {
        $this->isDeleteConfirmationOpen = false;
    }

    public function deleteConfirmed()
    {
        ClassSection::findOrFail($this->classSectionId)->delete();

        $this->dispatch('success', ['message' => 'تم تنفيذ العملية بنجاح']);
        $this->closeDeleteConfirmation();
    }

    public function sortBy($field)
    {
        $this->sortDirection = $this->sortField === $field
            ? ($this->sortDirection === 'asc' ? 'desc' : 'asc')
            : 'asc';

        $this->sortField = $field;
    }

    // New method to open calendar modal
    public function ShowCalendar($classSectionId)
    {
        $this->classPeriods = ClassPeriod::where('class_section_id', $classSectionId)->get();
        $this->isCalendarModalOpen = true;
        $this->dispatch('isCalendarModalOpen', ['classPeriods' => $this->classPeriods]);

    }

    public function closeCalendarModal()
    {
        $this->isCalendarModalOpen = false;
        $this->classPeriods = [];
    }

    public function render()
    {
        $classSections = ClassSection::query()
            ->where('name', 'like', '%' . $this->search . '%')
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(10);

        return view('livewire.class-sections.class-section-component', [
            'classSections' => $classSections,
            'classPeriods' => $this->classPeriods, // Pass classPeriods to the view

        ])->layout('layouts.admin.master');
    }
}
