<?php

namespace App\Livewire\Role;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;

class RoleComponent extends Component
{
    use WithPagination;
    public $isOpen = false; // Controls the modal visibility
    public $name,$search=''; // Holds the name input value
    public $permission_elements = []; // Holds the selected permission elements
    public $roleId; // Used to store the ID of the role being edited
    public $perPage = 10; // Pagination limit

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'permission_elements' => 'array',
        ];
    }

    public function render()
    {
        $roles = Role::with('permissions')
        ->where('name','like','%'.$this->search.'%')
        ->paginate($this->perPage); // Eager load permissions
        $permissions = Permission::with('permissionElements')->get(); // Fetch permissions with their elements

        return view('livewire.role.role-component', [
            'roles' => $roles,
            'permission' => $permissions,
        ])->layout('layouts.admin.master');
    }

    public function create()
    {
        $this->resetInputFields(); // Reset any existing values
        $this->isOpen = true; // Show the modal for creating a new role
        $this->roleId = null; // Set roleId to null for creating
    }

    public function edit($id)
    {
        $role = Role::findOrFail($id); // Fetch the role by ID

        $this->roleId = $role->id; // Store the ID for updating
        $this->name = $role->name; // Fill the name input
        $this->permission_elements = $role->permissions->pluck('id')->toArray(); // Fill selected permissions

        $this->isOpen = true; // Show the modal for editing
    }

    public function resetInputFields()
    {
        $this->reset(['name', 'permission_elements', 'roleId']); // Reset multiple properties at once
    }

    public function store()
    {
        $this->validate($this->rules()); // Validate input

        DB::transaction(function () {
            if ($this->roleId) {
                // Update the existing role
                $role = Role::findOrFail($this->roleId);
                $role->name = $this->name;
                $role->guard_name = $this->name;
                $role->save();
            } else {
                // Create a new role
                $role = Role::create([
                    'name' => $this->name,
                    'guard_name' => $this->name,
                ]);
            }
            $role->permissions()->sync($this->permission_elements); // Sync permissions
        });

        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']); // Dispatch success message
        $this->isOpen = false; // Close the modal
    }

    public function closeModal()
    {
        $this->isOpen = false; // Method to close modal
    }
}
