<?php

namespace App\Livewire\Schools;

use App\Imports\SchoolsImport;
use App\Models\EducationalArea;
use App\Models\School;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class SchoolComponent extends Component
{
    use WithPagination, WithFileUploads; // Use multiple traits in a single line

    public $isModalOpen = false, $file,$isDeleteConfirmationOpen,$isExcelModalOpen,$name, $schoolId, $educationalAreaId ,$status = 1,$perPage=10;
    public $search = '';
    public $sortField = 'name';
    public $sortDirection = 'asc';


    // Validation rules and messages
    protected $rules = [
        'name' => 'required|string|max:255',
        'educationalAreaId' => 'required|exists:educational_areas,id',
    ];

    protected $messages = [
        'name.required' => 'الاسم مطلوب.',
        'name.string' => 'الاسم يجب ان يكون نصا.',
        'name.max' => 'يجب ان يكون حجم النص اقل من 255 حرف.',
        'educationalAreaId.required' => 'المنطقة التعلمية مطلوبة',
        'educationalAreaId.exists' => 'المنطقة التعلمية غير موجودة',
    ];

    public function openExcelModal()
    {
        $this->isExcelModalOpen = true;
    }

    public function closeExcelModal()
    {
        $this->isExcelModalOpen = false;
    }

    public function uploadExcel()
    {
        $request->validate([
            'excel_file' => 'required|mimes:xlsx,xls',
        ]);

        // Handle the Excel file upload
        Excel::import(new SchoolsImport(), $this->file->getRealPath());

        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
        $this->closeExcelModal();
    }

    public function openModal($id = null)
    {
        $this->reset(['name', 'status', 'schoolId', 'educationalAreaId']);
        $this->isModalOpen = true;

        if ($id) {
            $this->loadSchool($id); // Load the school data
        }
    }

    protected function loadSchool($id)
    {
        $school = School::findOrFail($id);
        $this->educationalAreaId = $school->educational_area_id;
        $this->name = $school->name;
        $this->status = $school->status;
        $this->schoolId = $school->id;
    }

    public function closeModal()
    {
        $this->isModalOpen = false;
    }

    public function save()
    {
        $this->validate();
        School::updateOrCreate(
            ['id' => $this->schoolId],
            [
                'name' => $this->name,
                'educational_area_id' => $this->educationalAreaId,
                'status' => $this->status ? 1 : 0,
            ]
        );

        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
        $this->closeModal();
    }

    public function confirmDelete($id)
    {
        $this->schoolId = $id;
        $this->isDeleteConfirmationOpen = true;
    }

    public function closeDeleteConfirmation()
    {
        $this->isDeleteConfirmationOpen = false;
    }

    public function deleteConfirmed()
    {
        School::findOrFail($this->schoolId)->delete();
        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);

        $this->closeDeleteConfirmation();
    }

    public function updatingSearch()
    {
        $this->resetPage(); // Reset pagination when search is updated
    }

    public function sortBy($field)
    {
        // Toggle sort direction
        $this->sortDirection = ($this->sortField === $field && $this->sortDirection === 'asc') ? 'desc' : 'asc';
        $this->sortField = $field;
    }

    public function render()
    {
        $educationalAreas = EducationalArea::active()->get(); // Fetch active educational areas
        $schools = School::query()
            ->where('name', 'like', "%{$this->search}%")
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage); // Paginate results

        return view('livewire.schools.school-component', [
            'schools' => $schools,
            'educationalAreas' => $educationalAreas,
        ])->layout('layouts.admin.master');
    }
}
