<?php

namespace App\Livewire\Students;

use App\Imports\StudentImport;
use App\Models\ClassRoom;
use App\Models\ClassSection;
use App\Models\EducationalArea;
use App\Models\Governorate;
use App\Models\School;
use App\Models\Student;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class StudentComponent extends Component
{
    use WithFileUploads;
    use WithPagination;

    public $name, $id_number, $governorate_id, $educational_area_id, $school_id,$class_room_id,$class_section_id;
    public $isOpen = false;
    public $isDeleteConfirmationOpen=false;
    public $classSections = [];

    public $student_id;
    public $search = '';

    public $educational_areas = [];
    public $schools = [];

    protected $listeners = ['refresh' => '$refresh'];
    public $isExcelModalOpen = false;
    public $file;

    public function onGovernorateIdChange()
    {
        $this->educational_areas = EducationalArea::query()->where('governorate_id', $this->governorate_id)->get();
        $this->schools = []; // Reset schools when governorate changes
        $this->educational_area_id = null; // Reset educational area selection
        $this->school_id = null; // Reset school selection
    $this->class_room_id=null;

    }

    public function onEducationalAreaIdChange()
    {
        $this->schools = School::query()->where('educational_area_id', $this->educational_area_id)->get();
        $this->school_id = null; // Reset school selection
        $this->class_room_id=null;

    }

    public function create()
    {
        $this->resetInputFields();
        $this->isOpen = true;
    }

    public function onClassRoomChange()
    {
        $this->classSections = ClassSection::where('class_room_id', $this->class_room_id)->get();
        $this->class_section_id = null;  // Reset the section on class room change
    }
    public function edit($id)
    {
        $student = Student::findOrFail($id);
        $this->student_id = $id;
        $this->name = $student->name;
        $this->id_number = $student->id_number;
        $this->governorate_id = $student->governorate_id;
        $this->educational_area_id = $student->educational_area_id;
        $this->school_id = $student->school_id;
        // Load educational areas and schools based on the selected student
        $this->educational_areas = EducationalArea::where('governorate_id', $this->governorate_id)->get();
        $this->schools = School::where('educational_area_id', $this->educational_area_id)->get();

        $this->class_room_id=$student->class_room_id;
        $this->classSections = ClassSection::query()->where('class_room_id',$student->class_room_id)->get();
        $this->class_section_id=$student->class_section_id;

        $this->isOpen = true;
    }

    public function store()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'id_number' => 'required|string|unique:students,id_number,' . $this->student_id,
            'governorate_id' => 'required|exists:governorates,id',
            'educational_area_id' => 'required|exists:educational_areas,id',
            'school_id' => 'required|exists:schools,id',
        ]);

        $student = Student::find($this->student_id);

        $oldClassSectionId = $student ? $student->class_section_id : null;

// Update or create the student record
        $student = Student::updateOrCreate(
            ['id' => $this->student_id],
            [
               'name' => $this->name,
                'id_number' => $this->id_number,
                'governorate_id' => $this->governorate_id,
                'educational_area_id' => $this->educational_area_id,
                'school_id' => $this->school_id,
                'class_room_id' => $this->class_room_id,
                'class_section_id' => $this->class_section_id,
            ]
        );

// If class_section_id has changed, adjust the capacity accordingly
        if ($this->class_section_id && $oldClassSectionId !== $this->class_section_id) {
            // Increment capacity for the new section
            ClassSection::where('id', $this->class_section_id)->increment('current_capacity');

            // Decrement capacity for the old section, if applicable
            if ($oldClassSectionId) {

                ClassSection::where('id', $oldClassSectionId)->decrement('current_capacity');
            }
        }
        $this->resetInputFields();
        $this->isOpen = false;
        session()->flash('message', $this->student_id ? 'Student updated successfully.' : 'Student created successfully.');
        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
    }

//    public function delete($id)
//    {
//        Student::find($id)->delete();
//        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
//    }

    public function confirmDelete($id)
    {
        $this->student_id = $id;
        $this->isDeleteConfirmationOpen = true;
    }

    public function closeDeleteConfirmation()
    {
        $this->isDeleteConfirmationOpen = false;
    }

    public function deleteConfirmed()
    {
        Student::findOrFail($this->student_id)->delete();
        $this->closeDeleteConfirmation();
        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
    }

    private function resetInputFields()
    {
        $this->name = '';
        $this->id_number = '';
        $this->governorate_id = '';
        $this->educational_area_id = '';
        $this->school_id = '';
        $this->student_id = null;
        $this->educational_areas = [];
        $this->class_section_id=null;
        $this->schools = [];
    }
    public function updatingSearch()
    {
        $this->resetPage();
    }


    public function openExcelModal()
    {
        $this->isExcelModalOpen = true;
    }

    public function closeExcelModal()
    {
        $this->isExcelModalOpen = false;
    }

    public function uploadExcel()
    {
        $this->validate([
            'file' => 'required|file|mimes:xlsx,xls|max:2048',
        ]);
        Excel::import(new StudentImport(), $this->file->getRealPath());

        $this->dispatch('success', ['message' => 'تم تنفيد العملية بنجاح']);
        $this->closeExcelModal();
    }

    public function render()
    {
        $students = Student::query()
            ->where('is_active',1)
            ->where('name', 'like', "%{$this->search}%")
            ->
        paginate(10);
        $governorates = Governorate::all();
        $class_rooms = ClassRoom::all();


        $schools= $this->schools;
        $educational_areas= $this->educational_areas;
        return view('livewire.students.student-component', compact('class_rooms','students', 'governorates','educational_areas','schools'))->layout('layouts.admin.master');
    }

}
