<?php

namespace App\Livewire\Users;

use App\Models\Admin;
use App\Models\Role;
use Livewire\Component;
use Livewire\WithPagination;

class UserManagement extends Component
{
    use WithPagination;

    public $roles, $name, $email, $phone, $password, $search, $role_id, $user_id, $isOpen = false, $isDeleteConfirmationOpen = false;

    public $perPage=10;

    protected $rules = [
        'name' => 'required',
        'email' => 'required|email|unique:admins,email',
        'phone' => 'required|unique:admins,phone',
        'password' => 'sometimes|nullable|min:6',
        'role_id' => 'required',
    ];

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function openModal()
    {
        $this->resetInputFields();
        $this->isOpen = true;
    }

    public function closeModal()
    {
        $this->isOpen = false;
    }

    private function resetInputFields()
    {
        $this->reset(['name', 'email', 'phone', 'password', 'role_id', 'user_id']);
    }

    public function store()
    {
        $this->validate($this->getValidationRules());

        $user = Admin::updateOrCreate(
            ['id' => $this->user_id],
            [
                'name' => $this->name,
                'email' => $this->email,
                'phone' => $this->phone,
                'role_id' => $this->role_id,
            ]
        );

        // Only update password if provided
        if ($this->password) {
            $user->update(['password' => bcrypt($this->password)]);
        }

        $this->dispatch('success', ['message' => 'Operation successful']);
        $this->closeModal();
    }

    private function getValidationRules()
    {
        $rules = $this->rules;
        if ($this->user_id) {
            $rules['email'] .= ',' . $this->user_id; // Ignore current user on unique validation
            $rules['phone'] .= ',' . $this->user_id;
        }
        return $rules;
    }

    public function edit($id)
    {
        $this->openModal();

        $user = Admin::findOrFail($id);
        $this->fill($user->only(['name', 'email', 'phone', 'role_id']));
        $this->user_id = $id;
    }

    public function confirmDelete($id)
    {
        $this->user_id = $id;
        $this->isDeleteConfirmationOpen = true;
    }

    public function deleteConfirmed()
    {
        Admin::findOrFail($this->user_id)->delete();
        $this->dispatch('success', ['message' => 'User deleted successfully']);
        $this->isDeleteConfirmationOpen = false;
    }

    public function render()
    {
        $users = Admin::query()
            ->where(function($query) {
                $query->where('name', 'like', "%{$this->search}%")
                    ->orWhere('phone', 'like', "%{$this->search}%")
                    ->orWhere('email', 'like', "%{$this->search}%");
            })
            ->paginate($this->perPage);

        $this->roles = Role::all();
        return view('livewire.users.user-management', compact('users'))
            ->layout('layouts.admin.master');
    }
}
