<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Sanctum\HasApiTokens;

class Student extends Authenticatable
{
    use HasApiTokens, HasFactory, HasProfilePhoto, Notifiable;

    protected $guarded = [];

    // Relationships
    public function governorate()
    {
        return $this->belongsTo(Governorate::class, 'governorate_id');
    }

    public function educationalArea()
    {
        return $this->belongsTo(EducationalArea::class, 'educational_area_id');
    }

    public function school()
    {
        return $this->belongsTo(School::class, 'school_id','id');
    }

    public function classRooms()
    {
        return $this->belongsTo(ClassRoom::class, 'class_room_id')->withTrashed();
    }

    public function levels()
    {
        return $this->belongsTo(Level::class, 'level_id','id');
    }

    public function watchLessons()
    {
        return $this->belongsToMany(Lesson::class, 'student_watch_lessons', 'student_id', 'lesson_id')->withTimestamps();
    }

    public function markLessons()
    {
        return $this->belongsToMany(Lesson::class, 'student_mark_lessons', 'student_id', 'lesson_id')
            ->withPivot('mark')
            ->withTimestamps();
    }

    // Average calculations
    public function averageWatchedLessonsCount()
    {
        return $this->watchLessons()->count(); // Count of lessons watched by the student
    }

    public function avgWatchedLesson()
    {
        // Fetch all material IDs associated with the classroom
        $materialIds = Material::where('class_room_id', $this->class_room_id)->pluck('id')->toArray();
        $allLessonsCount = Lesson::whereIn('material_id', $materialIds)->count();
        $watchedLessonsCount = $this->watchLessons()->count();

        // Calculate and return the watched percentage, avoiding division by zero
        return $allLessonsCount > 0 ? round(($watchedLessonsCount / $allLessonsCount) * 100,2) : 0;
    }

    public function examResults()
    {
        return $this->hasMany(ExamResult::class, 'student_id', 'id');
    }

    public function avgExam()
    {
        $examResultIds = $this->examResults()->pluck('exam_id')->toArray();
        $totalExamResult = $this->examResults()->sum('result');
        $totalExamQuestions = Exam::whereIn('id', $examResultIds)->sum('question_count');

        return $totalExamQuestions > 0 ? round(($totalExamResult / $totalExamQuestions) * 100,2) : 0;
    }

    public function avgWatchedMaterialLesson($materialId)
{
    $allLessonsCount = Lesson::where('material_id', $materialId)->count();
    $watchedLessonsCount = $this->watchLessons()->where('material_id', $materialId)->count();
    return $allLessonsCount > 0 ? number_format(($watchedLessonsCount / $allLessonsCount) * 100, 2) : 0;
}
    public function avgMaterialExam($materialId)
    {
        $examResultIds = $this->examResults()->wherehas('exams',function($q)use ($materialId){
            $q->where('material_id', $materialId);
        })->pluck('exam_id')->toArray();
        $totalExamResult = $this->examResults()->wherehas('exams',function($q)use ($materialId){
            $q->where('material_id', $materialId);
        })->pluck('exam_id')
        ->sum('result');
        $totalExamQuestions = Exam::whereIn('id', $examResultIds)->sum('question_count');

        return $totalExamQuestions > 0 ? round(($totalExamResult / $totalExamQuestions) * 100,2) : 0;
    }

    public static function averageLessonsWatchedByAllStudents()
    {
        // Calculate average number of lessons watched across all students
        return round(self::withCount('watchLessons')->get()->avg('watch_lessons_count') * 100, 0);
    }

    public function averageMarks()
    {
        return round($this->markLessons()->avg('mark'), 2); // Return average mark from the pivot
    }

    public function generalAverageMark()
    {
        return round(($this->avgExam() + $this->averageMarks() + $this->avgWatchedLesson()) / 3,2);
    }

    public static function averageMarksForAllStudents()
    {
        // Calculate average marks from all students' marked lessons
        return round(self::with('markLessons')->get()->avg(function ($student) {
            return $student->markLessons->isNotEmpty() ? $student->markLessons->avg('pivot.mark') : 0;
        }), 2);
    }

    public function averageLessonsWatched($monthOffset = 0)
    {
        $startDate = Carbon::now()->subMonths($monthOffset)->startOfMonth();
        $endDate = Carbon::now()->subMonths($monthOffset)->endOfMonth();

        return $this->watchLessons()
            ->whereBetween('student_watch_lessons.created_at', [$startDate, $endDate])
            ->count();
    }

    public static function averageLessonsWatchedRate()
    {
        // Fetch current month and previous month lessons in one query for optimization
        $lessons = self::withCount(['watchLessons as lesson_count' => function ($query) {
            $query->whereBetween('student_watch_lessons.created_at', [
                Carbon::now()->subMonth()->startOfMonth(),
                Carbon::now()->endOfMonth()
            ]);
        }])->get();

        // Calculate averages
        $currentMonthAverage = $lessons->avg('lesson_count');
        $previousMonthAverage = $lessons->filter(function ($item) {
            return $item->created_at >= Carbon::now()->subMonth()->startOfMonth() &&
                   $item->created_at <= Carbon::now()->subMonth()->endOfMonth();
        })->avg('lesson_count');

        // Calculate the rate change
        if ($previousMonthAverage == 0) {
            return round(($currentMonthAverage * 100), 2); // Return as 100% increase if no previous data
        }

        return abs(round((($currentMonthAverage - $previousMonthAverage) / $previousMonthAverage) * 100, 2));
    }

    public static function averageLessonsWatchedForCurrentMonth()
    {
        $average = self::withCount(['watchLessons as current_month_count' => function ($query) {
            $query->whereBetween('student_watch_lessons.created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
        }])->get();

        // Return 0 if no data found
        return $average->avg('current_month_count') ?? 0;
    }

    public function avgMaterialMarks($materialId)
    {
        return round($this->markLessons()->wherehas('materials',function($q)use($materialId){
            $q->where('material_id',$materialId);
        })->avg('mark'), 2); // Return average mark from the pivot
    }
    public function avgGeneralMaterial($materialId)
    {

        return $this->avgMaterialExam($materialId)+$this->avgWatchedMaterialLesson($materialId)+$this->avgMaterialMarks($materialId);
        // Implement logic for average general material if needed
    }
}
